/* ***********************************************************************
   *                                                                     *
   * perfseq.c                                                           *
   * Christian Zurhorst, BID R&D, 30 Jul. 1998                           *
   *                                                                     *
   * performance sequencer functions                                     *
   *                                                                     *
   ***********************************************************************

*/


#ifdef BEST_FIRMWARE

#include <capitype.h>

#include <bios_malloc.h>

static void *malloc(b_int32 n)
{
  void *ptr;
  HWMalloc(&ptr, n);
  return ptr;
}


static void free(void *ptr)
{
  HWFree(ptr);
}


#else

#include <typedefs.h>

#endif

#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <perfseq.h>
#include <regconst.h>
#include <regx40.h>
#include <regx41.h>
#include <regx42.h>
#include <regx43.h>
#include <session.h>

extern char perfboard_err[127];


/* CAPI only calls */

#ifndef BEST_FIRMWARE

/*****************************************************************************/
/* This call initialises the descriptor table of the sequencer               */
/* of the selected measure                                                   */
/*****************************************************************************/

b_errtype EXPORT BestPerfSeqPropDefaultSet(b_handletype handle,
    b_int32 measure)
{
  b_errtype	err;
  b_int32 cmd_buf[2];
  b_int8 zw[4];
  /* first check if handle is good     */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* send command to performance option hardware                             */
  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    (void) BestLong2Stream(zw, &measure, 1UL);  /* never forget the measure  */
    err = BestBasicCommand(handle, CMD_PERFSEQ_PROPDEFAULTSET,
      zw, IN_PERFSEQ_PROPDEFAULTSET, NULL, NULL);
  }
  else
  {

    cmd_buf[0] = 9UL;           /* this is CLEAR_PERFSEQ_DESCR_TABLE   */
    cmd_buf[1] = measure;

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 8UL);
  }

  B_ERRETURN(err);

}

/*****************************************************************************/
/* This call sets all generic properties of the selected measure to default  */
/*****************************************************************************/

/* CONSIDER: why is this not implemented with dynamic capabilities? */
b_errtype EXPORT BestPerfGenPropDefaultSet(b_handletype handle,
    b_int32 measure)
{
  b_errtype	err;
  b_int32 cmd_buf[2];
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* send command to performance option hardware                             */
  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    B_ERRCHECK(BestPerfGenPropSet(handle, measure, B_PERFGEN_CTRC_PREL, 0UL));
    B_ERRETURN(BestPerfGenPropSet(handle, measure, B_PERFGEN_CAMODE, 0UL));
  }
  else
  {

    cmd_buf[0] = 23UL;          /* this is INIT_PERFGEN_PROP    */
    cmd_buf[1] = measure;

    B_ERRETURN(BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 8UL));
  }
}


#endif

/* CAPI + Firmware calls */


/*****************************************************************************/
/* This call sets a generic performance option property (integer based)      */
/* of the selected measure                                                   */
/*****************************************************************************/

b_errtype EXPORT BestPerfGenPropSet(b_handletype handle,
    b_int32 measure,
    b_perfgenproptype perfgenprop,
    b_int32 value)
{
  B_DECLARE_FUNCNAME ("BestPerfGenPropSet");
  b_errtype	err;
  b_int32 cmd_buf[4];           /* array yields command,property and    */
  b_int8 zw[IN_PERFSEQ_GENPROPSET];
  b_int8ptr bp;
  b_int32 zwl;
  /* property value                       */

  /* first check if handle is good and check license                    */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    bp = BestLong2Stream(zw, &measure, 1UL);  /* pack first the measure */
    zwl = (b_int32) perfgenprop;
    bp = BestLong2Stream(bp, &zwl, 1UL);  /* then take the property */
    (void) BestLong2Stream(bp, &value, 1UL);  /* and as last the value  */
    err = BestBasicCommand(handle, CMD_PERFSEQ_GENPROPSET, zw,
      IN_PERFSEQ_GENPROPSET, NULL, NULL);
  }
  else
  {
    /* send command, property parameter and data to performance option
     * hardware */

    cmd_buf[0] = 21UL;          /* this is defined as SET_PERFGEN_PROP    */
    cmd_buf[1] = measure;
    if (perfgenprop == B_PERFGEN_CTRC_PREL) /* fixed for compatibility reason */
      cmd_buf[2] = perfgenprop + B_PERFSEQ_GEN_PREL_OFFSET; /* enumerated property                   */
    else
      cmd_buf[2] = perfgenprop + B_PERFSEQ_GEN_CAMODE_OFFSET; /* enumerated property                   */
    cmd_buf[3] = value;         /* property value                         */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 16UL);

    /* if onboard error has been detected, receive error message string     */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err == B_E_OK) {
        B_ERRETURN(B_E_PERFBOARD_ERROR);
      }
    }
  }
  B_ERRETURN(err);

}


/*****************************************************************************/
/* This call returns the value of a generic property of the selected measure */
/*****************************************************************************/

b_errtype EXPORT BestPerfGenPropGet(b_handletype handle,
    b_int32 measure,
    b_perfgenproptype perfgenprop,
    b_int32 * value)
{
  B_DECLARE_FUNCNAME ("BestPerfGenPropGet");
  b_errtype	err;
  b_int32 cmd_buf[3];
  b_int32 recv_data;
  b_int8 zw[IN_PERF_GENPROP_GET];
  b_int8ptr bp;
  b_int32 zwl;
  b_int16 outsize;
  b_int8 outbuf[OUT_PERF_GENPROP_GET];
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    bp = BestLong2Stream(zw, &measure, 1UL);  /* pack first the measure */
    zwl = (b_int32) perfgenprop;
    (void) BestLong2Stream(bp, &zwl, 1UL);
    outsize = OUT_PERF_GENPROP_GET;
    err = BestBasicCommand(handle, CMD_PERF_GENPROP_GET,
      zw, IN_PERF_GENPROP_GET, outbuf, &outsize);
    (void) BestStream2Long(value, outbuf, 1UL);
  }
  else
  {

    /* send command, property parameter and data to performance option
     * hardware */

    cmd_buf[0] = 22UL;          /* this is defined as GET_PERFGEN_PROP    */
    cmd_buf[1] = measure;
    cmd_buf[2] = perfgenprop;   /* enumerated property                    */

    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 12UL));

    /* receive property value and return it                                  */
    err = BestBasicBlockRead(handle, PERF_COM, (b_int8ptr) & recv_data, 4UL);

    if (err == B_E_OK)
      *value = recv_data;
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call sets an integer based transient property in the sequencer       */
/* descriptor table of the selected measure                                  */
/*****************************************************************************/

b_errtype EXPORT BestPerfSeqTranPropSet(b_handletype handle,
    b_int32 measure,
    b_int32 transient,
    b_perfseqtranproptype perfseqtranprop,
    b_int32 value)
{
  B_DECLARE_FUNCNAME ("BestPerfSeqTranPropSet");
  b_errtype	err;
  b_int32 cmd_buf[5];           /* array holds command, measure,        */
  b_int8 zw[16];
  b_int8ptr bp;
  b_int32 zwl;
  /* transient, property, property value  */

  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* send command, property parameter and data to performance option hardware */

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    /* first copy the transient */
    bp = BestLong2Stream(zw, &measure, 1UL);  /* pack first the measure */
    bp = BestLong2Stream(bp, &transient, 1UL);  /* into the bytestream */
    zwl = (b_int32) perfseqtranprop;
    bp = BestLong2Stream(bp, &zwl, 1UL);  /* then take the property */
    (void) BestLong2Stream(bp, &value, 1UL);  /* and as the last take tye
                                               * value */
    err = BestBasicCommand(handle, CMD_PERFSEQ_TRANPROPSET,
      zw, IN_PERFSEQ_TRANPROPSET, NULL, NULL);
  }
  else
  {
    cmd_buf[0] = 17UL;          /* this is defined as SET_PERFSEQ_PROP    */
    cmd_buf[1] = measure;
    cmd_buf[2] = transient;
    cmd_buf[3] = perfseqtranprop + B_PERFSEQ_PROP_OFFSET; /* enumerated property */
    cmd_buf[4] = value;         /* property value                         */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 20UL);

    /* if onboard error has been detected, receive error message string        */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA, (b_int8ptr) perfboard_err, 127UL);

      if (err != B_E_OK)
      {
        B_ERRETURN(err);
      }
      else
      {
        B_ERRETURN(B_E_PERFBOARD_ERROR);
      }
    }
  }
  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call reads the value of an int based transient property from the     */
/* sequencer descriptor table of the selected measure                        */
/*****************************************************************************/

b_errtype EXPORT BestPerfSeqTranPropGet(b_handletype handle,
    b_int32 measure,
    b_int32 transient,
    b_perfseqtranproptype perfseqtranprop,
    b_int32 * value)
{
  B_DECLARE_FUNCNAME ("BestPerfSeqTranPropGet");
  b_errtype	err;
  b_int32 cmd_buf[4];           /* array holds command and property     */
  b_int32 recv_data;
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!BestIs2925(handle))
  {
    B_ERRETURN(B_E_ONLY_E2925A);
  }

  /* send command, property parameter and data to performance option hardware */

  cmd_buf[0] = 18UL;            /* this is defined as GET_PERFSEQ_PROP        */
  cmd_buf[1] = measure;
  cmd_buf[2] = transient;
  cmd_buf[3] = perfseqtranprop + B_PERFSEQ_PROP_OFFSET; /* enumerated property */

  B_ERRCHECK(BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 16UL));


  /* receive property value and return it                                    */
  err = BestBasicBlockRead(handle, PERF_COM, (b_int8ptr) & recv_data, 4UL);

  if (err == B_E_OK)
    *value = recv_data;

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call initialises a transient in the sequencer descriptor table       */
/* of the selected measure                                                   */
/*****************************************************************************/

b_errtype EXPORT BestPerfSeqTranPropDefaultSet(b_handletype handle,
    b_int32 measure,
    b_int32 transient)
{
  B_DECLARE_FUNCNAME ("BestPerfSeqTranPropDefaultSet");
  b_errtype	err;
  b_int32 cmd_buf[3];
  b_int8 zw[8];
  b_int8ptr bp;
  /* first check if handle is good     */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* send command to performance option hardware                             */

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    bp = BestLong2Stream(zw, &measure, 1UL);  /* pack first the measure */
    (void) BestLong2Stream(bp, &transient, 1UL);  /* and then the transient */
    err = BestBasicCommand(handle, CMD_PERFSEQ_TRANPROPDEFAULTSET,
      zw, IN_PERFSEQ_TRANPROPDEFAULTSET, NULL, NULL);

    /* make sure that the default for xcond is always '1' */
    if (err == B_E_OK)
      err = BestPerfSeqTranCondPropSet(handle, measure, transient,
        B_PERFSEQ_XCOND, "1");
  }
  else
  {
    cmd_buf[0] = 10UL;          /* this is INIT_ALL_PERFSEQ_TRANSIENT_PROP */
    cmd_buf[1] = measure;
    cmd_buf[2] = transient;

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 12UL);

    /* if onboard error has been detected, receive error message string     */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA, (b_int8ptr) perfboard_err, 127UL);
      if (err != B_E_OK)
      {
        B_ERRETURN(err);
      }
      else
      {
        B_ERRETURN(B_E_PERFBOARD_ERROR);
      }
    }
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call sets a string based transient property in the sequencer         */
/* descriptor table of the selected measure                                  */
/*****************************************************************************/

b_errtype EXPORT BestPerfSeqTranCondPropSet(b_handletype handle,
    b_int32 measure,
    b_int32 transient,
    b_perfseqtrancondproptype perfseqtrancondprop,
    b_charptrtype condition)
{
  B_DECLARE_FUNCNAME("BestPerfSeqTranCondPropSet [pstrancprpset]");

  b_errtype	err;
  char sync = '\1';             /* start of string marker                    */
  b_int32 cmd_buf[4];           /* command buffer                            */
  b_int8ptr zw;                 /* the buffer for the string */
  b_int8ptr bp;
  b_int32 zwl;
  b_int32 length;
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  B_FCT_PARAM_NULL_POINTER_CHK(condition);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */

    length = 13UL + (b_int32) strlen(condition);
    /* measure = 4 bytes, transient = 4 bytes, property = 4 bytes */
    /* then the string and the terminating 0 as the last byte */
    zw = (b_int8ptr) malloc((size_t) length);
    bp = BestLong2Stream(zw, &measure, 1UL);  /* pack first the measure */
    /* then copy the transient */
    bp = BestLong2Stream(bp, &transient, 1UL);  /* into the bytestream */
    zwl = (b_int32) perfseqtrancondprop;
    bp = BestLong2Stream(bp, &zwl, 1UL);  /* then take the property */
    (void) BestByteCopy(bp, (b_int8ptr) condition, strlen(condition) + 1UL);
    /* and the last to copy is the condition */

    err = BestBasicCommandVariable(handle, CMD_PERFSEQ_TRANCONDPROPSET,
      zw, (b_int16) length, NULL, NULL);

    free(zw);                   /* free the allocated memory */
  }
  else
  {

    cmd_buf[0] = 17UL;          /* this is SET_PERFSEQ_PROP */
    cmd_buf[1] = measure;
    cmd_buf[2] = transient;
    cmd_buf[3] = perfseqtrancondprop + B_PERFSEQ_CONDITION_OFFSET;

    /* first transfer condition string via the performance data port        */

    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_DATA, (b_int8ptr) & sync, 1UL));
    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_DATA,
        (b_int8ptr) condition, (b_int32) strlen(condition) + 1UL));

    /* then send command and property parameter to performance hardware      */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 16UL);

    /* if onboard error has been detected, receive error message string     */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err != B_E_OK)
      {
        B_ERRETURN(err);
      }
      else
      {
        B_ERRETURN(B_E_PERFBOARD_ERROR);
      }
    }
  }
  B_ERRETURN(err);

}


/*****************************************************************************/
/* This call reads a string property of a transient from the sequencer       */
/* descriptor table of the selected measure. Note that                       */
/* this call returns the +parsed+ string property, that is, a 32bit function */
/* map                                                                       */
/*****************************************************************************/

b_errtype EXPORT BestPerfSeqTranCondPropGet(b_handletype handle,
    b_int32 measure,
    b_int32 transient,
    b_perfseqtrancondproptype perfseqtrancondprop,
    b_int32 * value)
{
  B_DECLARE_FUNCNAME ("BestPerfSeqTranCondPropGet");
  b_errtype	err;
  b_int32 cmd_buf[4];           /* array holds command and property     */
  b_int32 recv_data;
  /* first check if handle is good                                          */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!BestIs2925(handle))
  {
    B_ERRETURN(B_E_ONLY_E2925A);
  }

  /* send command, property parameter and data to performance option hardware */

  cmd_buf[0] = 18UL;            /* this is defined as GET_PERFSEQ_PROP      */
  cmd_buf[1] = measure;
  cmd_buf[2] = transient;
  cmd_buf[3] = perfseqtrancondprop + B_PERFSEQ_CONDITION_OFFSET;
  /* enumerated property              */

  B_ERRCHECK(BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 16UL));

  /* receive property value and return it                                    */

  err = BestBasicBlockRead(handle, PERF_COM, (b_int8ptr) & recv_data, 4UL);

  if (err == B_E_OK)
    *value = recv_data;

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call programs the  sequencer memory of the performance counter       */
/* of the selected measure                                                   */
/*****************************************************************************/

b_errtype EXPORT BestPerfSeqProg(b_handletype handle,
    b_int32 measure)
{
  B_DECLARE_FUNCNAME("BestPerfSeqProg [psprog]");

  b_errtype	err;
  b_int8 zw[4];
  b_int32 cmd_buf[2];
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  /* send command to performance option hardware                             */

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    B_FCT_PARAM_CHK(1, measure > 7);

    (void) BestLong2Stream(zw, &measure, 1UL);  /* pack the measure - we need
                                                 * it */
    err = BestBasicCommand(handle, CMD_PERF_SEQPROG,
      zw, IN_PERF_SEQPROG, NULL, NULL);
  }
  else
  {

    cmd_buf[0] = 11UL;          /* this is PROG_PERFSEQ_MEMORY       */
    cmd_buf[1] = measure;

    B_FCT_PARAM_CHK(1, measure > 1UL);

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 8UL);

    /* if onboard error has been detected, receive error message string      */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err == B_E_OK) {
        B_ERRETURN(B_E_PERFBOARD_ERROR);
      }
    }
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call starts all performance counters                                 */
/*****************************************************************************/

b_errtype EXPORT BestPerfRun(b_handletype handle)
{
  B_DECLARE_FUNCNAME ("BestPerfRun");
  b_errtype	err;
  b_int32 cmd_type;
  /* first check if handle is good     */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    err = BestBasicCommand(handle, CMD_PERF_RUN, NULL, IN_PERF_RUN, NULL, NULL);
  }
  else
  {
    /* send command to performance option hardware                          */

    cmd_type = 12UL;            /* this is PERFCNT_RUN           */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) & cmd_type, 4UL);
  }

  B_ERRETURN(err);

}


/*****************************************************************************/
/* This call stops all performance counters                                  */
/*****************************************************************************/

b_errtype EXPORT BestPerfStop(b_handletype handle)
{
  B_DECLARE_FUNCNAME ("BestPerfStop");
  b_errtype	err;
  b_int32 cmd_type;
  /* first check if handle is good     */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    err = BestBasicCommand(handle, CMD_PERF_STOP, IN_PERF_STOP, 0, NULL, NULL);
  }
  else
  {
    /* send command to performance option hardware                          */

    cmd_type = 13UL;            /* this is PERFCNT_STOP         */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) & cmd_type, 4UL);
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call returns the status of the performance measure                   */
/*****************************************************************************/

b_errtype EXPORT BestPerfStatusGet(b_handletype handle,
    b_int32 measure,
    b_int32 * value)
{
  B_DECLARE_FUNCNAME ("BestPerfStatusGet");
  b_errtype	err;
  b_int32 cmd_buf[2];
  b_int32 recv_data;
  b_int8 in_zw[IN_PERF_STATUS];
  b_int8 out_zw[OUT_PERF_STATUS];
  b_int16 addr;
  /* first check if handle is good                                          */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    addr = OUT_PERF_STATUS;
    (void) BestLong2Stream(in_zw, &measure, 1UL); /* measure into the
                                                   * bytestream */
    err = BestBasicCommand(handle, CMD_PERF_STAT_READ,
      in_zw, IN_PERF_STATUS, out_zw, &addr);
    (void) BestStream2Long(value, out_zw, 1UL);
  }
  else
  {
    /* send command,property parameter and data to performance option
     * hardware */

    cmd_buf[0] = 15UL;          /* this is defined as PERFSTATUSREG_GET    */
    cmd_buf[1] = measure;

    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 8UL));

    /* receive register value and return it */

    B_ERRCHECK(BestBasicBlockRead(handle, PERF_COM, (b_int8ptr) & recv_data, 4UL));

    *value = recv_data;
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call updates the performance counter registers and restarts all      */
/* counters. Note that for a valid register contents this function must be   */
/* called in intervals chosen so that the performance counter wont overflow  */
/*****************************************************************************/

b_errtype EXPORT BestPerfUpdate(b_handletype handle)
{
  B_DECLARE_FUNCNAME ("BestPerfUpdate");
  b_errtype	err;
  b_int32 cmd_type;
  /* first check if handle is good     */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    err = BestBasicCommand(handle, CMD_PERF_UPDATE, NULL, IN_PERF_UPDATE,
      NULL, NULL);
  }
  else
  {
    /* send command to performance option hardware                          */

    cmd_type = 14UL;            /* this is PERFMEASURE_UPDATE   */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) & cmd_type, 4UL);
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call returns the contents of the performance counter registers       */
/*****************************************************************************/

b_errtype EXPORT BestPerfCtrRead(b_handletype handle,
    b_int32 measure,
    b_int32 counter_id,
    b_int32 * value)
{
  B_DECLARE_FUNCNAME ("BestPerfCtrRead");
  b_errtype	err;
  b_int32 cmd_buf[3];
  b_int32 recv_data;
  b_int8 in_zw[IN_PERF_CTR];
  b_int8 out_zw[OUT_PERF_CTR];
  b_int16 addr;
  b_int8ptr bp;
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    addr = OUT_PERF_CTR;
    bp = BestLong2Stream(in_zw, &measure, 1UL); /* measure into the
                                                 * bytestream */
    bp = BestLong2Stream(bp, &counter_id, 1UL); /* counter id also ... */
    err = BestBasicCommand(handle, CMD_PERF_CTR_READ,
      in_zw, IN_PERF_CTR, out_zw, &addr);
    (void) BestStream2Long(value, out_zw, 1UL);
  }
  else
  {
    /* send command, property parameter and data to performance option
     * hardware */

    cmd_buf[0] = 16UL;          /* this is defined as PERFCOUNTREG_GET   */
    cmd_buf[1] = measure;
    cmd_buf[2] = counter_id;

    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 12UL));

    /* receive property value and return it                                 */

    err = BestBasicBlockRead(handle, PERF_COM, (b_int8ptr) & recv_data, 4UL);

    *value = recv_data;

    /* if onboard error has been detected, receive error message string      */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err != B_E_OK)
      {
        B_ERRETURN(err);
      }
      else
      {
        B_ERRETURN(B_E_PERFBOARD_ERROR);
      }
    }
  }
  B_ERRETURN(err);

}


b_errtype EXPORT BestAboutPerfBoardGet(b_handletype handle,
    b_int32 * info)
{
  B_DECLARE_FUNCNAME ("BestAboutPerfBoardGet");
  b_errtype err;
  
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  handle = handle;              /* this avoids a compiler warning */
  *info = 1997;
  B_ERRETURN(B_E_OK);
}


